#!/system/bin/sh
DEBUG=${DEBUG:-false}
$DEBUG && set -x || set +x

ui_print() {
    echo "$@"
}

require_new_ksu() {
    ui_print "**********************************"
    ui_print " Please install KernelSU v0.6.6+! "
    ui_print "**********************************"
    exit 1
}

umount_mirrors() {
    [ -d "$ORIGDIR" ] || return 0
    for mirror in "$ORIGDIR"/*; do
        umount -l "$mirror" 2>/dev/null
    done
    rm -rf "$ORIGDIR" 2>/dev/null
    mount -o ro,remount "$MAGISKTMP"
}

cleanup() {
    $KSU && umount_mirrors
    rm -rf "$MODPATH/common" "$MODPATH/install.zip" "$MODPATH/DEMONIC.rs" \
           "$MODPATH/DEMONIC.c" "$MODPATH/changelog.md" "$MODPATH/README.md" "$MODPATH/customize.sh" 2>/dev/null
}

abort() {
    ui_print "Abort: Cleaning up and exiting..."
    rm -rf "$MODPATH" 2>/dev/null
    cleanup
    rm -rf "$TMPDIR" 2>/dev/null
    exit 1
}

trap 'abort' INT TERM

device_check() {
    local opts type="device"
    opts=$(getopt -o dm -- "$@") || return 1
    eval set -- "$opts"
    
    while true; do
        case "$1" in
            -d) type="device"; shift ;;
            -m) type="manufacturer"; shift ;;
            --) shift; break ;;
            *) abort "Invalid device_check argument: $1! Aborting!" ;;
        esac
    done
    
    local expected prop
    expected=$(echo "$1" | tr '[:upper:]' '[:lower:]')
    
    for file in /system /vendor /odm /product; do
        [ -f "$file/build.prop" ] || continue
        
        for prefix in "ro.product.$type" "ro.build.$type" \
                      "ro.product.vendor.$type" "ro.vendor.product.$type"; do
            prop=$(sed -n "s/^${prefix}=//p" "$file/build.prop" 2>/dev/null | head -n1 | tr '[:upper:]' '[:lower:]')
            [ "$prop" = "$expected" ] && return 0
        done
        
        if [ "$type" = "device" ]; then
            prop=$(sed -n 's/^ro.build.product=//p' "$file/build.prop" 2>/dev/null | head -n1 | tr '[:upper:]' '[:lower:]')
            [ "$prop" = "$expected" ] && return 0
        fi
    done
    return 1
}

cp_ch() {
    local opts BAK=true UBAK=true FOL=false
    opts=$(getopt -o nr -- "$@") || return 1
    eval set -- "$opts"
    
    while true; do
        case "$1" in
            -n) UBAK=false; shift ;;
            -r) FOL=true; shift ;;
            --) shift; break ;;
            *) abort "Invalid cp_ch argument: $1! Aborting!" ;;
        esac
    done

    local SRC="$1" DEST="$2" PERM=${3:-0644} file_list
    file_list="$SRC"
    [ "$FOL" = true ] && file_list=$(find "$SRC" -type f 2>/dev/null)
    
    case "$DEST" in
        "$TMPDIR"/*|"$MODULEROOT"/*|"$NVBASE/modules/$MODID"/*) BAK=false ;;
    esac

    for file in $file_list; do
        local target
        if [ "$FOL" = true ]; then
            if [ "$(basename "$SRC")" = "$(basename "$DEST")" ]; then
                target=$(echo "$file" | sed "s|$SRC|$DEST|")
            else
                target=$(echo "$file" | sed "s|$SRC|$DEST/$(basename "$SRC")|")
            fi
        else
            target="${DEST%/}/$(basename "$SRC")"
        fi

        if $BAK && $UBAK; then
            { [ -z "$(grep "$target$" "$INFO" 2>/dev/null)" ] && echo "$target" >> "$INFO"; } || true
            { [ -f "$target" ] && [ ! -f "${target}~" ] && \
              mv -f "$target" "${target}~" && echo "${target}~" >> "$INFO"; } || true
        elif $BAK; then
            [ -z "$(grep "$target$" "$INFO" 2>/dev/null)" ] && echo "$target" >> "$INFO"
        fi
        
        install -D -m "$PERM" "$file" "$target"
    done
}

install_script() {
    local INPATH args="$1"
    shift
    
    case "$args" in
        -b)
            if $KSU; then
                INPATH="$NVBASE/boot-completed.d"
            else
                INPATH="$SERVICED"
                sed -i "1i while [ \"\$(getprop sys.boot_completed)\" != \"1\" ]; do sleep 1; done; sleep 3" "$1"
            fi
            ;;
        -l) INPATH="$SERVICED" ;;
        -p) INPATH="$POSTFSDATAD" ;;
        *) INPATH="$SERVICED" ;;
    esac

    shebang="#!/system/bin/sh"
    grep -q "^$shebang" "$1" || sed -i "1i $shebang" "$1"
    
    for var in MODPATH LIBDIR MODID INFO MODDIR; do
        case $var in
            MODPATH) val="$NVBASE/modules/$MODID" ;;
            MODDIR)  val='${0%/*}' ;;
            *)       val=$(eval echo \$$var) ;;
        esac
        sed -i "1a $var=\"$val\"" "$1"
    done

    case "$1" in
        "$MODPATH/post-fs-data.sh"|"$MODPATH/service.sh"|"$MODPATH/uninstall.sh")
            sed -i "s|^MODPATH=.*|MODPATH=\$MODDIR|" "$1"
            ;;
        "$MODPATH/boot-completed.sh")
            $KSU && sed -i "s|^MODPATH=.*|MODPATH=\$MODDIR|" "$1" || {
                cp_ch -n "$1" "$INPATH/$MODID-$(basename "$1")" 0755
                rm -f "$1"
            }
            ;;
        *) cp_ch -n "$1" "$INPATH/$(basename "$1")" 0755 ;;
    esac
}

prop_process() {
    sed -i -e '/^#/d' -e '/^$/d' "$1"
    [ -f "$MODPATH/system.prop" ] || mktouch "$MODPATH/system.prop"
    cat "$1" >> "$MODPATH/system.prop"
}

mount_mirrors() {
    mount -o rw,remount "$MAGISKTMP" || abort "Failed to remount $MAGISKTMP as RW"
    mkdir -p "$ORIGDIR/system"
    
    if $SYSTEM_ROOT; then
        mkdir -p "$ORIGDIR/system_root"
        mount -o ro / "$ORIGDIR/system_root" || abort "Failed to mount system_root"
        mount -o bind "$ORIGDIR/system_root/system" "$ORIGDIR/system"
    else
        mount -o ro /system "$ORIGDIR/system" || abort "Failed to mount system"
    fi
    
    for part in /vendor $PARTITIONS; do
        [ -d "$part" ] || continue
        mkdir -p "$ORIGDIR$part"
        mount -o ro "$part" "$ORIGDIR$part"
    done
}

[ -z "$MINAPI" ] || { [ "$API" -lt "$MINAPI" ] && abort "! System API $API < minimum API $MINAPI"; }
[ -z "$MAXAPI" ] || { [ "$API" -gt "$MAXAPI" ] && abort "! System API $API > maximum API $MAXAPI"; }

[ -z "$KSU" ] && KSU=false
$KSU && { [ "$KSU_VER_CODE" -lt 11184 ] && require_new_ksu; }

[ -z "$ARCH32" ] && ARCH32="$(echo "$ABI32" | cut -c-3)"
[ "$API" -lt 26 ] && DYNLIB=false
[ -z "$DYNLIB" ] && DYNLIB=false
[ -z "$PARTOVER" ] && PARTOVER=false

INFO="$NVBASE/modules/.$MODID-files"
if $KSU; then
    MAGISKTMP="/mnt"
    ORIGDIR="$MAGISKTMP/mirror"
    mount_mirrors
else
    MAGISKTMP="$(magisk --path 2>/dev/null || echo /sbin)"
    ORIGDIR="$MAGISKTMP/.magisk/mirror"
fi

LIBPATCH=$($DYNLIB && echo "/vendor" || echo "/system")
LIBDIR="/system$($DYNLIB && echo "/vendor")"

EXTRAPART=false
$KSU || case "$(magisk --version)" in
    *delta*) EXTRAPART=true ;;
    *) $PARTOVER || unset PARTITIONS ;;
esac

if ! $BOOTMODE; then
    ui_print "- Recovery mode detected: Uninstalling"
    touch "$MODPATH/remove"
    [ -s "$INFO" ] && install_script "$MODPATH/uninstall.sh" || rm -f "$INFO"
    recovery_cleanup
    cleanup
    rm -rf "$NVBASE/modules_update/$MODID" "$TMPDIR" 2>/dev/null
    exit 0
fi

ui_print "- Extracting module files"
unzip -o "$ZIPFILE" -x 'META-INF/*' 'common/functions.sh' -d "$MODPATH" >&2

if [ -f "$MODPATH/common/addon.tar.xz" ]; then
    tar -xf "$MODPATH/common/addon.tar.xz" -C "$MODPATH/common" 2>/dev/null
    if [ -n "$(ls -A "$MODPATH/common/addon/"*/install.sh 2>/dev/null)" ]; then
        ui_print " "
        ui_print "- Running Addons -"
        for addon in "$MODPATH"/common/addon/*/install.sh; do
            ui_print "- Running $(basename "$(dirname "$addon")")"
            . "$addon"
        done
    fi
fi

ui_print "- Removing old files 🚮"
if [ -f "$INFO" ]; then
    while IFS= read -r line; do
        [ "${line:~ -1}" = "~" ] && continue
        [ -f "${line}~" ] && mv -f "${line}~" "$line" || rm -f "$line"
        while [ -n "$line" ]; do
            line=$(dirname "$line")
            [ -d "$line" ] && rm -rf "$line" 2>/dev/null
        done
    done < "$INFO"
    rm -f "$INFO"
fi

ui_print "- Installing for $ARCH SDK $API"
[ -f "$MODPATH/common/install.sh" ] && . "$MODPATH/common/install.sh"

find "$MODPATH" -type \( -name "*.sh" -o -name "*.prop" -o -name "*.rule" \) | while read -r file; do
    sed -i -e '/^#/d' -e '/^$/d' "$file"
    [ -z "$(tail -1 "$file")" ] && echo >> "$file"
    
    case "$file" in
        "$MODPATH/post-fs-data.sh"|"$MODPATH/service.sh"|"$MODPATH/uninstall.sh")
            install_script "$file"
            ;;
        "$MODPATH/boot-completed.sh")
            install_script -b "$file"
            ;;
    esac
done

$IS64BIT || find "$MODPATH/system" -type d -name "lib64" -exec rm -rf {} +

[ -d "/system/priv-app" ] || mv -f "$MODPATH/system/priv-app" "$MODPATH/system/app"
[ -d "/system/xbin" ] || mv -f "$MODPATH/system/xbin" "$MODPATH/system/bin"

if $DYNLIB; then
    find "$MODPATH/system/lib"* -type f | while read -r file; do
        case "$file" in
            */modules/*) continue ;;
            *) 
                vendor_file="${file/system/vendor}"
                mkdir -p "$(dirname "$vendor_file")"
                mv -f "$file" "$vendor_file"
                ;;
        esac
    done
    find "$MODPATH/system/lib"* -type d -empty -delete
fi

ui_print "- Setting permissions"
set_perm_recursive "$MODPATH" 0 0 0755 0644
set_perm "$MODPATH/system/vendor/etc/flp.conf" 0 0 0644
set_perm "$MODPATH/system/vendor/etc/gps.conf" 0 0 0644
set_perm "$MODPATH/system/vendor/etc/gnss.conf" 0 0 0644
set_perm "$MODPATH/system/vendor/etc/gps/gps.xml" 0 0 0644

find "$MODPATH" -type d | while read -r dir; do
    case "$dir" in
        */vendor)           set_perm_recursive "$dir" 0 0 0755 0644 u:object_r:vendor_file:s0 ;;
        */vendor/app)        set_perm_recursive "$dir" 0 0 0755 0644 u:object_r:vendor_app_file:s0 ;;
        */vendor/overlay)   set_perm_recursive "$dir" 0 0 0755 0644 u:object_r:vendor_overlay_file:s0 ;;
        */vendor/etc|*/etc)  set_perm_recursive "$dir" 0 2000 0755 0644 u:object_r:vendor_configs_file:s0 ;;
    esac
done

find "$MODPATH/system/vendor" "$MODPATH/vendor" -name "*.apk" | while read -r apk; do
    chcon u:object_r:vendor_app_file:s0 "$apk"
done

cleanup