#!/system/bin/sh
# Do NOT assume where your module will be located.
# ALWAYS use $MODDIR if you need to know where this script
# and module is placed.
# This will make sure your module will still work
# if Magisk change its mount point in the future
MODDIR=${0%/*}

# ----------------- VARIABLES -----------------
ps_ret="$(ps -Ao pid,args)"
GED_PATH="/sys/module/ged/parameters"
GED_PATH2="/sys/kernel/debug/ged/hal"
GPUF_PATH="/proc/gpufreq"
GPUF_PATHV2="/proc/gpufreqv2"
PVR_PATH="/sys/module/pvrsrvkm/parameters"
PVR_PATH2="/sys/kernel/debug/pvr/apphint"
ADRENO_PATH="/sys/class/kgsl/kgsl-3d0"
ADRENO_PATH2="/sys/kernel/debug/kgsl/kgsl-3d0/profiling"
ADRENO_PATH3="/sys/module/adreno_idler/parameters"
KERNEL_FPSGO_PATH="/sys/kernel/debug/fpsgo/common"
MALI_PATH="/proc/mali"
PLATFORM_GPU_PATH="/sys/devices/platform/gpu"
GPUFREQ_TRACING_PATH="/sys/kernel/debug/tracing/events/mtk_events/gpu_freq"

# ----------------- HELPER FUNCTIONS -----------------
log() {
    echo "$1"
}

wait_until_boot_completed() {
    while [ "$(getprop sys.boot_completed)" != "1" ]; do sleep 3; done
    while [ "$(dumpsys window policy | grep mInputRestricted=true)" != "" ]; do sleep 0.1; done
    while [ ! -d "/sdcard/Android" ]; do sleep 1; done
}

mask_val() {
    touch /data/local/tmp/mount_mask
    for p in $2; do
        if [ -f "$p" ]; then
            umount "$p"
            chmod 0666 "$p"
            echo "$1" >"$p"
            mount --bind /data/local/tmp/mount_mask "$p"
        fi
    done
}

write_val() {
    local file="$1"
    local value="$2"
    if [ -e "$file" ]; then
        chmod +w "$file" 2>/dev/null
        echo "$value" > "$file" && log "Write : $file → $value" || log "Failed to Write : $file"
    fi
}

change_task_cgroup() {
    local task="$1"
    local group="$2"
    local class="$3"
    for pid in $(echo "$ps_ret" | grep -iE "$task" | awk '{print $1}'); do
        for tid in /proc/"$pid"/task/*; do
            temp_tid="${tid##*/}"
            if [ -e "$tid" ]; then
                echo "$temp_tid" > "/dev/${class}/${group}/tasks" 2>/dev/null
            fi
        done
    done
}

change_task_nice() {
    local task="$1"
    local nice_val="$2"
    for pid in $(echo "$ps_ret" | grep -iE "$task" | awk '{print $1}'); do
        for tid in /proc/"$pid"/task/*; do
            temp_tid="${tid##*/}"
            if [ -e "$tid" ]; then
                renice -n "$nice_val" -p "$temp_tid" 2>/dev/null
                ionice -c 1 -n 0 -p "$tid" 2>/dev/null
            fi
        done
    done
}

send_notification() {
    # Notify user of optimization completion
    su -lp 2000 -c "cmd notification post -S bigtext -t 'Celestial-Render-FlowX👾' tag 'Status : Optimization Completed!'" >/dev/null 2>&1
}

# ----------------- OPTIMIZATION SECTIONS -----------------
optimize_gpu_temperature() {
    # Adjust GPU and DDR temperature thresholds ( @Bias_khaliq )
    for THERMAL in /sys/class/thermal/thermal_zone*/type; do
        if grep -E "gpu|ddr" "$THERMAL" > /dev/null; then
          for ZONE in "${THERMAL%/*}"/trip_point_*_temp; do
            CURRENT_TEMP=$(cat "$ZONE")
            if [ "$CURRENT_TEMP" -lt "90000" ]; then
              echo "95000" > "$ZONE"
            fi
          done
        fi
    done
}

additional_gpu_settings() {
    # Optimize GPU parameters via GED driver
    if [ -d "$GED_PATH" ]; then
        write_val "$GED_PATH/gpu_cust_boost_freq" "2000000"
        write_val "$GED_PATH/gpu_cust_upbound_freq" "2000000"
        write_val "$GED_PATH/ged_smart_boost" "1000"
        write_val "$GED_PATH/gpu_bottom_freq" "800000"
        write_val "$GED_PATH/boost_upper_bound" "100"
        write_val "$GED_PATH/gx_dfps" "$(dumpsys display | grep -oE 'fps=[0-9]+' | grep -oE '[0-9]+' | sort -nr | head -n 1)"
        write_val "$GED_PATH/g_gpu_timer_based_emu" "1"
        write_val "$GED_PATH/boost_gpu_enable" "1"
        write_val "$GED_PATH/ged_boost_enable" "1"
        write_val "$GED_PATH/enable_gpu_boost" "1"
        write_val "$GED_PATH/gx_game_mode" "1"
        write_val "$GED_PATH/gx_boost_on" "1"
        write_val "$GED_PATH/boost_amp" "1"
        write_val "$GED_PATH/gx_3D_benchmark_on" "1"
        write_val "$GED_PATH/is_GED_KPI_enabled" "1"
        write_val "$GED_PATH/gpu_dvfs_enable" "1"
        write_val "$GED_PATH/ged_monitor_3D_fence_disable" "1"
        write_val "$GED_PATH/ged_monitor_3D_fence_debug" "0"
        write_val "$GED_PATH/ged_log_perf_trace_enable" "0"
        write_val "$GED_PATH/ged_log_trace_enable" "0"
        write_val "$GED_PATH/gpu_bw_err_debug" "0"
        write_val "$GED_PATH/gx_frc_mode" "0"
        write_val "$GED_PATH/gpu_idle" "0"
        write_val "$GED_PATH/gpu_debug_enable" "0"
    else
        echo "Unknown $GED_PATH path. Skipping optimization."
    fi
    
    
    # Additional kernel-ged GPU optimizations
    if [ -d "$GED_PATH2" ]; then
         write_val "$GED_PATH2/gpu_boost_level" "2"
    else
        echo "Unknown $GED_PATH2 path. Skipping optimization."
    fi
    
    # Additional GPU settings for MediaTek ( @Bias_khaliq )
    if [ -d "$PLATFORM_GPU_PATH" ]; then
         write_val "$PLATFORM_GPU_PATH/dvfs_enable" "1"
         write_val "$PLATFORM_GPU_PATH/gpu_busy" "1"
    else
        echo "Unknown $GED_PATH2 path. Skipping optimization."
    fi
}

optimize_gpu_frequency() {
    # Optimize GPU frequency configurations
    gpu_freq="$(cat $GPUF_PATH/gpufreq_opp_dump | grep -o 'freq = [0-9]*' | sed 's/freq = //' | sort -nr | head -n 1)"
    if [ -d "$GPUF_PATH" ]; then
        for i in $(seq 0 8); do
            write_val "$GPUF_PATH/limit_table" "$i 0 0"
        done
        # Activate mid-range
        for i in 2 3 4 5; do
            write_val "$GPUF_PATH/limit_table" "$i 1 1"
        done
        write_val "$GPUF_PATH/gpufreq_limited_thermal_ignore" "1"
        write_val "$GPUF_PATH/gpufreq_limited_oc_ignore" "1"
        write_val "$GPUF_PATH/gpufreq_limited_low_batt_volume_ignore" "1"
        write_val "$GPUF_PATH/gpufreq_limited_low_batt_volt_ignore" "1"
        write_val "$GPUF_PATH/gpufreq_fixed_freq_volt" "0"
        write_val "$GPUF_PATH/gpufreq_opp_freq" "$gpu_freq"
        write_val "$GPUF_PATH/gpufreq_opp_stress_test" "0"
        write_val "$GPUF_PATH/gpufreq_power_dump" "0"
        write_val "$GPUF_PATH/gpufreq_power_limited" "0"
    else
        echo "Unknown $GPUF_PATH path. Skipping optimization."
    fi

    # Optimize GPU frequency v2 configurations (Matt Yang)（吟惋兮改)
    gpu_freq="$(cat $GPUF_PATH2/gpu_working_opp_table | awk '{print $3}' | sed 's/,//g' | sort -nr | head -n 1)"
	gpu_volt="$(cat $GPUF_PATH2/gpu_working_opp_table | awk -v freq="$freq" '$0 ~ freq {gsub(/.*, volt: /, ""); gsub(/,.*/, ""); print}')"
    if [ -d "$GPUF_PATH2" ]; then
		write_val "$GPUF_PATH2/fix_custom_freq_volt" "${gpu_freq} ${gpu_volt}"
        write_val "$GPUF_PATH2/aging_mode" "disable"
        for i in $(seq 0 10); do
            write_val "$GPUF_PATH2/limit_table" "$i 0 0"
        done
        # Enable only levels 3–6 (mid-high stable range)
        for i in 3 4 5 6; do
            write_val "$GPUF_PATH2/limit_table" "$i 1 1"
        done
    else
        echo "Unknown $GPUF_PATH2 path. Skipping optimization."
    fi
}

optimize_pvr_settings() {
    # Adjust PowerVR settings for performance
    if [ -d "$PVR_PATH" ]; then
        write_val "$PVR_PATH/gpu_power" "2"
        write_val "$PVR_PATH/HTBufferSizeInKB" "512"
        write_val "$PVR_PATH/DisableClockGating" "1"
        write_val "$PVR_PATH/EmuMaxFreq" "3"
        write_val "$PVR_PATH/EnableFWContextSwitch" "1"
        write_val "$PVR_PATH/gPVRDebugLevel" "0"
        write_val "$PVR_PATH/gpu_dvfs_enable" "0"
    else
        echo "Unknown $PVR_PATH path. Skipping optimization."
    fi

    # Additional settings power vr apphint
    if [ -d "$PVR_PATH2" ]; then
        write_val "$PVR_PATH2/CacheOpConfig" "1"
        write_val "$PVR_PATH2/CacheOpUMKMThresholdSize" "512"
        write_val "$PVR_PATH2/EnableFTraceGPU" "0"
        write_val "$PVR_PATH2/HTBOperationMode" "2"
        write_val "$PVR_PATH2/TimeCorrClock" "1"
        write_val "$PVR_PATH2/0/DisableFEDLogging" "1"
        write_val "$PVR_PATH2/0/EnableAPM" "0"
    else
        echo "Unknown $PVR_PATH2 path. Skipping optimization."
    fi
}

optimize_adreno_driver() {
    # Additional kgsl settings to stabilize the gpu (Matt Yang)（吟惋兮改)
    if [ -d "$ADRENO_PATH" ]; then
        PWRLVL=$(($(cat $ADRENO_PATH/num_pwrlevels) - 1))
        mask_val "$PWRLVL" "$ADRENO_PATH/default_pwrlevel"
        mask_val "$PWRLVL" "$ADRENO_PATH/min_pwrlevel"
        mask_val "1" "$ADRENO_PATH/force_bus_on"
        mask_val "1" "$ADRENO_PATH/force_clk_on"
        mask_val "1" "$ADRENO_PATH/force_no_nap"
        mask_val "1" "$ADRENO_PATH/force_rail_on"
        mask_val "0" "$ADRENO_PATH/thermal_pwrlevel"
        mask_val "0" "$ADRENO_PATH/throttling"
        mask_val "0" "$ADRENO_PATH/max_pwrlevel"
        mask_val "0" "$ADRENO_PATH/fsync_enable"
        mask_val "0" "$ADRENO_PATH/vsync_enable"
    else
        echo "Unknown $ADRENO_PATH path. Skipping optimization."
    fi
}

optimize_mali_driver() {
    # Mali GPU-specific optimizations ( @Bias_khaliq )
    if [ -d "$MALI_PATH" ]; then
         write_val "$MALI_PATH/dvfs_enable" "1"
         write_val "$MALI_PATH/max_clock" "550000"
         write_val "$MALI_PATH/min_clock" "100000"
    else
        echo "Unknown $MALI_PATH path. Skipping optimization."
    fi
}

optimize_task_cgroup_nice() {
    # thx to (Matt Yang)（吟惋兮改) edited by @Kzuyoo
    change_task_cgroup "surfaceflinger" "top-app" "cpuset"
    change_task_cgroup "surfaceflinger" "top-app" "stune"
    change_task_nice "surfaceflinger" "-10"

    change_task_cgroup "system_server" "foreground" "cpuset"
    change_task_cgroup "system_server" "foreground" "stune"

    change_task_cgroup "netd|allocator" "foreground" "cpuset"

    change_task_cgroup "hardware.media.c2|vendor.mediatek.hardware" "background" "cpuset"
    change_task_cgroup "hardware.media.c2|vendor.mediatek.hardware" "background" "stune"

    change_task_cgroup "aal_sof|kfps|dsp_send_thread|vdec_ipi_recv|mtk_drm_disp_id|disp_feature|hif_thread|main_thread|rx_thread|ged_" "background" "cpuset"

    change_task_cgroup "pp_event|crtc_" "background" "cpuset"

    change_task_cgroup "android.hardware.graphics.composer" "top-app" "cpuset"
    change_task_cgroup "android.hardware.graphics.composer" "top-app" "stune"
    change_task_nice "android.hardware.graphics.composer" "-15"
}

final_optimize_gpu() {
    # Additional kernel-fpsgo GPU optimizations
    if [ -d "$KERNEL_FPSGO_PATH" ]; then
        write_val "$KERNEL_FPSGO_PATH/gpu_block_boost" "60 120 1"
    else
        echo "Unknown $KERNEL_FPSGO_PATH path. Skipping optimization."
    fi
    
    # disable pvr tracing
    for pvrtracing in $(find /sys/kernel/debug/tracing/events/pvr_fence -name 'enable'); do
        if [ -d "/sys/kernel/debug/tracing/events/pvr_fence" ]; then
            write_val "$pvrtracing" "0"
        fi
    done
        
    # disable gpu tracing for mtk
    write_val "$GPUFREQ_TRACING_PATH/enable" "0"
   
    # disable kgsl profiling
    write_val "$ADRENO_PATH2/enable" "0"
    
    # disable adreno idler
    write_val "$ADRENO_PATH3/adreno_idler_active" "0"
    
   # Disable auto voltage scaling for mtk
    write_val "$GPU_FREQ_PATH/gpufreq_aging_enable" "0"
}

cleanup_memory() {
    # Clean up memory and cache
     write_val "/proc/sys/vm/drop_caches" "3"
     write_val "/proc/sys/vm/compact_memory" "1"
}

# ----------------- MAIN EXECUTION -----------------
main() {
    wait_until_boot_completed
    optimize_gpu_temperature
    additional_gpu_settings
    optimize_gpu_frequency
    optimize_pvr_settings
    optimize_adreno_driver
    optimize_mali_driver
    optimize_task_cgroup_nice
    final_optimize_gpu
    cleanup_memory
}

# Main Execution & Exit script successfully
sync && main && send_notification && exit 0

# This script will be executed in late_start service mode
