// script.js — ChronoFPS AxManager
// Module (KernelSU) alapú végrehajtás
import { exec, toast } from 'https://cdn.jsdelivr.net/npm/kernelsu@1.0.6/+esm';

// --- Utility
function log(msg){
  const el=document.getElementById('logOutput');
  const now=new Date().toLocaleTimeString();
  el.textContent = `[${now}] ${msg}\n` + el.textContent;
}

function disableBtn(id, state=true){
  const b=document.getElementById(id);
  if(b) b.disabled=state;
}

async function runCommands(commands){
  // commands: array of shell command strings
  for(const c of commands){
    try{
      log(`> ${c}`);
      await exec(c + ' >/dev/null 2>&1');
    }catch(e){
      log(`ERR: ${String(e).slice(0,200)}`);
    }
  }
}

// --- Package list (az általad megadott csomagnevek)
const GAME_PACKAGES = [
  "age.of.civilizations2.jakowski.lukasz",
  "com.bushiroad.en.bangdreamgbp",
  "com.ChillyRoom.DungeonShooter",
  "com.chillyroom.soulknightprequel",
  "com.Flanne.MinutesTillDawn.roguelike.shooting.gp",
  "com.GameCoaster.ProtectDungeon",
  "com.HoYoverse.hkrpgoversea",
  "com.RoamingStar.BlueArchive",
  "com.Shooter.ModernWarships",
  "com.Sunborn.SnqxExilium",
  "com.YoStar.AetherGazer",
  "com.YoStarEN.Arknights",
  "com.YoStarEN.MahjongSoul",
  "com.YoStarJP.MajSoul",
  "com.YostarJP.BlueArchive",
  "com.ZeroCastleGameStudio.StrikeBusterPrototype",
  "com.ZeroCastleGameStudioINTL.StrikeBusterPrototype",
  "com.activision.callofduty.shooter",
  "com.activision.callofduty.warzone",
  "com.albiononline",
  "com.aligames.kuang.kybc",
  "com.autumn.skullgirls",
  "com.axlebolt.standoff2",
  "com.bandainamcoent.imas_millionlive_theaterdays",
  "com.bandainamcoent.sao",
  "com.bandainamcoent.shinycolorsprism",
  "com.bhvr.deadbydaylight",
  "com.bilibili.azurlane",
  "com.bilibili.deadcells.mobile",
  "com.bilibili.fatego",
  "com.bilibili.star.bili",
  "com.bilibili.warmsnow",
  "com.bilibiligame.heglgp",
  "com.bingkolo.kleins.cn",
  "com.blizzard.diablo.immortal",
  "com.bushiroad.d4dj",
  "com.bushiroad.lovelive.schoolidolfestival2",
  "com.carxtech.sr",
  "com.citra.emu",
  "com.cnvcs.xiangqi",
  "com.criticalforceentertainment.criticalops",
  "com.dena.a12026801",
  "com.denachina.g13002010",
  "com.dfjz.moba",
  "com.dgames.g15002002",
  "com.dolphinemu.dolphinemu",
  "com.dragonli.projectsnow.lhm",
  "com.dts.freefiremax",
  "com.dts.freefireth",
  "com.dts.freefireadv",
  "com.ea.gp.apexlegendsmobilefps",
  "com.ea.gp.fifamobile",
  "com.epicgames.fortnite",
  "com.fantablade.icey",
  "com.firsttouchgames.dls7",
  "com.gabama.monopostolite",
  "com.gameloft.android.ANMP.GloftA9HM",
  "com.gameloft.android.ANMP.GloftMVHM",
  "com.garena.game.codm",
  "com.garena.game.kgid",
  "com.garena.game.kgtw",
  "com.garena.game.kgvn",
  "com.guyou.deadstrike",
  "com.halo.windf.hero",
  "com.heavenburnsred",
  "com.hermes.j1game",
  "com.hottapkgs.hotta",
  "com.hypergryph.arknights",
  "com.ignm.raspberrymash.jp",
  "com.ilongyuan.implosion",
  "com.jacksparrow.jpmajiang",
  "com.je.supersus",
  "com.kakaogames.eversoul",
  "com.kakaogames.wdfp",
  "com.kog.grandchaseglobal",
  "com.komoe.kmumamusumegp",
  "com.kurogame.haru",
  "com.kurogame.haru.bilibili",
  "com.leiting.wf",
  "com.levelinfinite.hotta.gp",
  "com.levelinfinite.sgameGlobal",
  "com.levelinfinite.sgameGlobal.midaspay",
  "com.lilithgames.hgame.cn",
  "com.linegames.sl",
  "com.madfingergames.legends",
  "com.miHoYo.GenshinImpact",
  "com.miHoYo.Yuanshen",
  "com.miHoYo.bh3",
  "com.miHoYo.bh3global",
  "com.miHoYo.bh3oversea",
  "com.miHoYo.bh3oversea_vn",
  "com.miHoYo.enterprise.NGHSoD",
  "com.miHoYo.hkrpg",
  "com.miHoYo.ys",
  "com.miraclegames.farlight84",
  "com.mobile.legends",
  "com.mobilelegends.hwag",
  "com.mojang.hostilegg",
  "com.mojang.minecraftpe",
  "com.mojang.minecraftpe.patch",
  "com.nanostudios.games.twenty.minutes",
  "com.netease.AVALON",
  "com.netease.EVE",
  "com.netease.aceracer",
  "com.netease.dfjs",
  "com.netease.dwrg",
  "com.netease.eve.en",
  "com.netease.frxyna",
  "com.netease.g93na",
  "com.netease.h75na",
  "com.netease.jddsaef",
  "com.netease.lglr",
  "com.netease.ma84",
  "com.netease.moba",
  "com.netease.mrzh",
  "com.netease.newspike",
  "com.netease.nshm",
  "com.netease.onmyoji",
  "com.netease.party",
  "com.netease.partyglobal",
  "com.netease.race",
  "com.netease.racerna",
  "com.netease.sky",
  "com.netease.soulofhunter",
  "com.netease.tj",
  "com.netease.tom",
  "com.netease.wotb",
  "com.netease.x19",
  "com.netease.yhtj",
  "com.nexon.bluearchive",
  "com.nexon.kartdrift",
  "com.nianticlabs.monsterhunter",
  "com.olzhass.carparking.multyplayer",
  "com.pearlabyss.blackdesertm.gl",
  "com.pinkcore.tkfm",
  "com.playdigious.deadcells.mobile",
  "com.proximabeta.mf.uamo",
  "com.proximabeta.nikke",
  "com.prpr.musedash",
  "com.pubg.imobile",
  "com.pubg.krmobile",
  "com.pubg.newstate",
  "com.pwrd.hotta.laohu",
  "com.pwrd.huanta",
  "com.r2games.myhero.bilibili",
  "com.rayark.implosion",
  "com.rayark.sdorica",
  "com.rekoo.pubgm",
  "com.riotgames.league.teamfighttactics",
  "com.riotgames.league.teamfighttacticsvn",
  "com.riotgames.league.wildrift",
  "com.roblox.client",
  "com.rockstargames.gtasa",
  "com.sega.ColorfulStage.en",
  "com.sega.pjsekai",
  "com.shangyoo.neon",
  "com.shenlan.m.reverse1999",
  "com.smokoko.race",
  "com.sofunny.Sausage",
  "com.soulgamechst.majsoul",
  "com.sprduck.garena.vn",
  "com.stove.epic7.google",
  "com.supercell.brawlstars",
  "com.supercell.clashofclans",
  "com.supercell.clashroyale",
  "com.sy.dldlhsdj",
  "com.t2ksports.nba2k20and",
  "com.tencent.KiHan",
  "com.tencent.ig",
  "com.tencent.jkchess",
  "com.tencent.lolm",
  "com.tencent.mf.uam",
  "com.tencent.tmgp.WePop",
  "com.tencent.tmgp.bh3",
  "com.tencent.tmgp.cf",
  "com.tencent.tmgp.cod",
  "com.tencent.tmgp.dfjs",
  "com.tencent.tmgp.ffom",
  "com.tencent.tmgp.gnyx",
  "com.tencent.tmgp.kr.codm",
  "com.tencent.tmgp.pubgmhd",
  "com.tencent.tmgp.sgame",
  "com.tencent.tmgp.speedmobile",
  "com.tencent.tmgp.wuxia",
  "com.tencent.tmgp.yys.zqb",
  "com.tencent.toaa",
  "com.tgc.sky.android",
  "com.the10tons.dysmantle",
  "com.ubisoft.rainbowsixmobile.r6.fps.pvp.shooter",
  "com.unity.mmd",
  "com.valvesoftware.cswgsm",
  "com.valvesoftware.source",
  "com.vng.mlbbvn",
  "com.vng.pubgmobile",
  "com.vng.speedvn",
  "com.xd.TLglobal",
  "com.xd.rotaeno.googleplay",
  "com.xd.rotaeno.tapcn",
  "com.xindong.torchlight",
  "com.yongshi.tenojo",
  "com.zlongame.mhmnz",
  "com.ztgame.bob",
  "com.zy.wqmt.cn",
  "jp.co.craftegg.band",
  "jp.konami.pesam",
  "net.kdt.pojavlaunch",
  "net.wargaming.wot.blitz",
  "org.mm.jr",
  "org.vita3k.emulator",
  "pro.archiemeng.waifu2x",
  "skyline.emu",
  "skynet.cputhrottlingtest",
  "org.yuzu.yuzu_emu",
  "id.rj01117883.liomeko",
  "xyz.aethersx2.android",
  "com.pwrd.opmwsea",
  "com.kurogame.gplay.punishing.grayraven.en",
  "com.FosFenes.Sonolus",
  "com.devsisters.ck",
  "ro.alyn_sampmobile.game",
  "com.kakaogames.gdts",
  "com.Shooter.ModernWarship",
  "com.supercell.hayday",
  "com.nekki.shadowfight",
  "com.nekki.shadowfight3",
  "com.gamedevltd.wwh",
  "com.netease.ma100asia",
  "com.gryphline.exastris.gp",
  "org.sudachi.sudachi_emu.ea",
  "com.netmarble.sololv",
  "com.netease.g78na.gb",
  "com.pubg.imobile",
  "com.bandainamcoent.opbrww",
  "com.proximabeta.mf.uamo",
  "com.kurogame.wutheringwaves.global",
  "com.HoYoverse.Nap"
];

// --- Base CONFIG template (a te általad megadott nagy CONFIG string).
const BASE_CONFIG = `mode=2,fps={FPS},skiavk=1,vsync=0,frameRateCap={FPS},refreshRate={FPS},minFrameRate={FPS},maxFrameRate={FPS},frameRateLimit={FPS},targetFrameRate={FPS},lowLatencyMode=1,angle=1,gpuBoost=1,cpuBoost=1,sustainedPerformanceMode=1,adaptiveFrameRate=0,renderAheadLimit=2,touchBoost=1,frameStabilization=1,frameRateSmoothing=1,gpuGovernor=performance,cpuGovernor=performance,thermalThrottling=0,surfaceFlingerTripleBuffering=1,gameMode=1,thermalHeadroomBoost=1,priorityBoost=1;mode=3,fps={FPS},skiavk=1,vsync=0,frameRateCap={FPS},refreshRate={FPS},minFrameRate={FPS},maxFrameRate={FPS},frameRateLimit={FPS},targetFrameRate={FPS},lowLatencyMode=1,angle=1,gpuBoost=1,cpuBoost=1,sustainedPerformanceMode=1,adaptiveFrameRate=0,renderAheadLimit=2,touchBoost=1,frameStabilization=1,frameRateSmoothing=1,gpuGovernor=performance,cpuGovernor=performance,thermalThrottling=0,surfaceFlingerTripleBuffering=1,gameMode=1,thermalHeadroomBoost=1,priorityBoost=1`;

function buildConfig(fps){
  return BASE_CONFIG.replace(/{FPS}/g, String(fps));
}

// --- Handlers

async function applyTargetFps(){
  const val = document.getElementById('targetFps').value;
  const config = buildConfig(val);
  const commands = [];

  // device_config put loop
  for(const pkg of GAME_PACKAGES){
    commands.push(`cmd device_config put game_overlay ${pkg} "${config}"`);
  }

  // setprop three debug refresh props (as in your code)
  commands.push(`setprop debug.refresh_rate.min_fps ${val}`);
  commands.push(`setprop debug.refresh_rate.max_fps ${val}`);
  commands.push(`setprop debug.refresh_rate.peak_fps ${val}`);

  disableBtn('applyTargetFpsBtn', true);
  log(`Applying Target FPS ${val} to ${GAME_PACKAGES.length} packages...`);
  await runCommands(commands);
  disableBtn('applyTargetFpsBtn', false);
  toast(`Target FPS ${val} applied`);
  log('Target FPS applied.');
}

async function applyRefreshRate(){
  const val = document.getElementById('refreshRate').value;
  const cmds = [
    `settings put system peak_refresh_rate ${val}`,
    `settings put system user_refresh_rate ${val}`,
    `settings put system min_refresh_rate ${val}`,
    `settings put system thermal_limit_refresh_rate ${val}`,
    `settings put system miui_refresh_rate ${val}`,
    `settings put secure user_refresh_rate ${val}`,
    `settings put secure max_refresh_rate ${val}`,
    `settings put secure miui_refresh_rate ${val}`,
    `settings put global preferred_refresh_rate ${val}`,
    `settings put global user_preferred_refresh_rate ${val}`
  ];
  disableBtn('applyRefreshRateBtn', true);
  await runCommands(cmds);
  disableBtn('applyRefreshRateBtn', false);
  toast(`Refresh rate set to ${val}Hz`);
  log(`Refresh rate set to ${val}Hz`);
}

async function applyPerformanceMode(){
  const val=document.getElementById('performanceMode').value;
  disableBtn('applyPerfModeBtn', true);
  if(val==='perf'){
    const cmds=[
      `settings put system POWER_PERFORMANCE_MODE_OPEN 1`,
      `settings put system multicore_packet_scheduler 1`,
      `settings put system high_performance_mode_on 1`,
      `settings put system sem_performance_mode 4`,
      `settings put system speed_mode 1`,
      `settings put secure high_priority 1`,
      `settings put secure speed_mode_enable 1`,
      `settings put global sem_enchanced_cpu_responsiveness 1`,
      `settings put global cached_apps_freezer "enabled"`,
      `settings put global restricted_device_performance "0,0"`,
      `settings put global adaptive_battery_management_enabled 0`,
      `settings put global game_auto_temperature_control 0`,
      `settings put system perf_shielder_SF 0`,
      `settings put system perf_shielder_RTMODE 1`,
      `settings put system perf_shielder_GESTURE 0`,
      `settings put system perf_shielder_smartpower 0`,
      `settings put system POWER_SAVE_MODE_OPEN 0`,
      `settings put global automatic_power_save_mode 0`,
      `settings put global low_power 0`,
      `setprop debug.performance.tuning 1`,
      `setprop debug.sf.hw 1`,
      `setprop debug.egl.hw 1`,
      `setprop debug.egl.buffercount 2`,
      `setprop debug.gralloc.enable_fb_ubwc 1`,
      `setprop debug.gralloc.gfx_ubwc_disable 0`,
      `setprop debug.gralloc.map_fb_memory 0`,
      `setprop debug.sf.multithreaded_present 1`,
      `setprop debug.sf.enable_layer_caching true`,
      `setprop debug.sf.predict_hwc_composition_strategy 1`,
      `setprop debug.hwui.force_draw_frame true`,
      `setprop debug.hwui.skip_empty_damage true`,
      `setprop debug.hwui.use_gpu_pixel_buffers true`,
      `cmd power set-fixed-performance-mode-enabled true`,
      `cmd power set-adaptive-power-saver-enabled false`,
      `cmd power set-mode 0`
    ];
    await runCommands(cmds);
    toast('Performance mode activated');
    log('Performance mode applied');
  }else{
    const cmds=[
      `settings put system POWER_PERFORMANCE_MODE_OPEN 0`,
      `settings put system multicore_packet_scheduler 0`,
      `settings put system high_performance_mode_on 0`,
      `settings put system sem_performance_mode 1`,
      `settings put system speed_mode 0`,
      `settings put secure high_priority 0`,
      `settings put secure speed_mode_enable 0`,
      `settings put global sem_enchanced_cpu_responsiveness 0`,
      `settings put global cached_apps_freezer "disabled"`,
      `settings put global restricted_device_performance "0,0"`,
      `settings put global adaptive_battery_management_enabled 1`,
      `settings put global game_auto_temperature_control 1`,
      `settings put system perf_shielder_SF 1`,
      `settings put system perf_shielder_RTMODE 0`,
      `settings put system perf_shielder_GESTURE 1`,
      `settings put system perf_shielder_smartpower 1`,
      `settings put system POWER_SAVE_MODE_OPEN 1`,
      `settings put global automatic_power_save_mode 1`,
      `settings put global low_power 1`,    
      `setprop debug.performance.tuning 0`,
      `setprop debug.sf.hw 0`,
      `setprop debug.egl.hw 0`,
      `setprop debug.egl.buffercount 1`,
      `setprop debug.gralloc.enable_fb_ubwc 0`,
      `setprop debug.gralloc.gfx_ubwc_disable 1`,
      `setprop debug.gralloc.map_fb_memory 1`,
      `setprop debug.sf.multithreaded_present 0`,
      `setprop debug.sf.enable_layer_caching false`,
      `setprop debug.sf.predict_hwc_composition_strategy 0`,
      `setprop debug.hwui.force_draw_frame false`,
      `setprop debug.hwui.skip_empty_damage false`,
      `setprop debug.hwui.use_gpu_pixel_buffers false`,      
      `cmd power set-fixed-performance-mode-enabled false`,
      `cmd power set-adaptive-power-saver-enabled true`,
      `cmd power set-mode 1`
    ];
    await runCommands(cmds);
    toast('Power Save mode activated');
    log('Power Save mode applied');
  }
  disableBtn('applyPerfModeBtn', false);
}

async function applyRenderer(){
  const val=document.getElementById('renderer').value;
  const cmds=[
    `setprop debug.hwui.renderer ${val}`,
    `setprop debug.renderengine.backend ${val}`
  ];
  disableBtn('applyRendererBtn', true);
  await runCommands(cmds);
  disableBtn('applyRendererBtn', false);
  toast(`Renderer set to ${val}`);
  log(`Renderer set to ${val}`);
}

async function applyVsync(){
  const val=document.getElementById('vsync').value;
  let cmds=[];
  if(val==='enabled'){
    cmds=[
      `setprop debug.gr.swapinterval 1`,
      `setprop debug.sf.swapinterval 1`,
      `setprop debug.egl.swapinterval 1`,
      `setprop debug.gl.swapinterval 1`,
      `setprop debug.hwui.disable_vsync false`,
      `setprop debug.hwc.force_gpu_vsync 1`,
      `setprop debug.sf.no_hw_vsync 0`,
      `setprop debug.hwc.fakevsync 1`,
      `setprop debug.logvsync 1`,
      `setprop debug.choreographer.vsync true`,
      `setprop debug.cpurend.vsync true`
    ];
  }else{
    cmds=[
      `setprop debug.gr.swapinterval 0`,
      `setprop debug.sf.swapinterval 0`,
      `setprop debug.egl.swapinterval 0`,
      `setprop debug.gl.swapinterval 0`,
      `setprop debug.hwui.disable_vsync true`,
      `setprop debug.hwc.force_gpu_vsync 0`,
      `setprop debug.sf.no_hw_vsync 1`,
      `setprop debug.hwc.fakevsync 0`,
      `setprop debug.logvsync 0`,
      `setprop debug.choreographer.vsync false`,
      `setprop debug.cpurend.vsync false`
    ];
  }
  disableBtn('applyVsyncBtn', true);
  await runCommands(cmds);
  disableBtn('applyVsyncBtn', false);
  toast(`VSync ${val}`);
  log(`VSync ${val}`);
}

async function applyCpuPerf(){
  const raw=document.getElementById('cpuPerf').value;
  const val=parseInt(raw.replace('%',''));
  const cmds=[
    `setprop debug.hwui.target_cpu_time_percent ${val}`,
    `settings put global activity_manager_constants power_check_max_cpu_1=${val},power_check_max_cpu_2=${val},power_check_max_cpu_3=${val},power_check_max_cpu_4=${val}`
  ];
  disableBtn('applyCpuPerfBtn', true);
  await runCommands(cmds);
  disableBtn('applyCpuPerfBtn', false);
  toast(`CPU Performance ${val}%`);
  log(`CPU Performance ${val}%`);
}

async function applyGpuRenderer(){
  const val=document.getElementById('gpuRenderer').value;
  const cmds=[`setprop debug.composition.type ${val}`];
  disableBtn('applyGpuRendererBtn', true);
  await runCommands(cmds);
  disableBtn('applyGpuRendererBtn', false);
  toast(`GPU Renderer ${val}`);
  log(`GPU Renderer ${val}`);
}

async function applyThermal(){
  const val=document.getElementById('thermal').value; // 0..6
  const cmd = `cmd thermalservice override-status ${val}`;
  disableBtn('applyThermalBtn', true);
  await runCommands([cmd]);
  disableBtn('applyThermalBtn', false);
  toast(`Thermal override ${val}`);
  log(`Thermal override ${val}`);
}

async function applyFramePriority(){
  const val=document.getElementById('framePriority').value;
  if(val==='enabled'){
    const cmds=[
      `setprop debug.sf.use_frame_rate_priority 1`,
      `setprop debug.sf.frame_rate_multiple_threshold 0`,
      `settings put secure match_content_frame_rate 0`,
      `settings put system framepredict_enable 1`,
      `cmd display set-match-content-frame-rate-pref 0`
    ];
    disableBtn('applyFramePriorityBtn', true);
    await runCommands(cmds);
    disableBtn('applyFramePriorityBtn', false);
    toast('Frame Rate Priority enabled');
    log('Frame Rate Priority enabled');
  }else{
    const cmds=[
      `setprop debug.sf.use_frame_rate_priority 0`,
      `setprop debug.sf.frame_rate_multiple_threshold 1`,
      `settings put secure match_content_frame_rate 1`,
      `settings put system framepredict_enable 0`,
      `cmd display set-match-content-frame-rate-pref 1`
    ];
    disableBtn('applyFramePriorityBtn', true);
    await runCommands(cmds);
    disableBtn('applyFramePriorityBtn', false);
    toast('Frame Rate Priority disabled');
    log('Frame Rate Priority disabled');
  }
}

// --- Wire up DOM on load
window.addEventListener('load', ()=>{
  document.getElementById('applyTargetFpsBtn').addEventListener('click', applyTargetFps);
  document.getElementById('applyRefreshRateBtn').addEventListener('click', applyRefreshRate);
  document.getElementById('applyPerfModeBtn').addEventListener('click', applyPerformanceMode);
  document.getElementById('applyRendererBtn').addEventListener('click', applyRenderer);
  document.getElementById('applyVsyncBtn').addEventListener('click', applyVsync);
  document.getElementById('applyCpuPerfBtn').addEventListener('click', applyCpuPerf);
  document.getElementById('applyGpuRendererBtn').addEventListener('click', applyGpuRenderer);
  document.getElementById('applyThermalBtn').addEventListener('click', applyThermal);
  document.getElementById('applyFramePriorityBtn').addEventListener('click', applyFramePriority);

  // date bar updater
  const dateBar = document.getElementById('dateBar');
  setInterval(()=>{ if(dateBar) dateBar.textContent = new Date().toLocaleString(); },1000);
});