#!/system/bin/sh
# Combined MTK AI Engine: Voltage Guardian + CPU Limiter

# --- 1. CONFIGURATION ---
LOG_FILE="/sdcard/MTK_AI_Engine/MTK_AI_Engine.log"
TARGET_TEMP_FILE="/sys/class/power_supply/battery/temp"
CPU_SYS="/sys/devices/system/cpu"
CPU_LIMIT_PERC=30

EEM_PATHS="
/proc/eem/EEM_DET_B/eem_offset
/proc/eem/EEM_DET_BL/eem_offset
/proc/eem/EEM_DET_L/eem_offset
/proc/eem/EEM_DET_CCI/eem_offset
/proc/eemg/EEMG_DET_GPU/eemg_offset
/proc/eemg/EEMG_DET_GPU_HI/eemg_offset
"

# --- 2. HELPER FUNCTIONS ---
log_msg() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
    [ "$(wc -l < $LOG_FILE)" -gt 200 ] &&
        tail -n 200 "$LOG_FILE" > "$LOG_FILE.tmp" &&
        mv "$LOG_FILE.tmp" "$LOG_FILE"
}

get_batt_temp() {
    if [ -f "$TARGET_TEMP_FILE" ]; then
        # Converts standard thermal millidegrees to Celsius
        echo $(( $(cat "$TARGET_TEMP_FILE") / 10 ))
    else
        echo 0
    fi
}

# --- 3. DYNAMIC VOLTAGE GUARDIAN ---
temp=$(get_batt_temp)

case $temp in
    36) offset="-1"  ;;
    37) offset="-2"  ;;
    38) offset="-3"  ;;
    39) offset="-4"  ;;
    40) offset="-5"  ;;
    41) offset="-6"  ;;
    42) offset="-7"  ;;
    43) offset="-8"  ;;
    44) offset="-9"  ;;
    45) offset="-10" ;;
    46) offset="-11" ;;
    47|48|49|50) offset="-12" ;;
    *)  offset="0"   ;;
esac

log_msg "THERMAL: Temp=${temp}C | Action: Applying Offset ${offset}mV"

for path in $EEM_PATHS; do
    if [ -f "$path" ]; then
        echo "$offset" > "$path"
    fi
done

# --- 4. CPU FREQUENCY LIMITER ---
log_msg "FREQ: Setting CPU limit to ${CPU_LIMIT_PERC}%..."

for policy in $CPU_SYS/cpufreq/policy*; do
    [ -d "$policy" ] || continue
    
    # Allow minimum scale down
    if [ -w "$policy/scaling_min_freq" ]; then
        echo 0 > "$policy/scaling_min_freq"
    fi

    # Calculate 30% of max hardware frequency
    if [ -f "$policy/cpuinfo_max_freq" ]; then
        max_val=$(cat "$policy/cpuinfo_max_freq")
        target_val=$(( max_val * CPU_LIMIT_PERC / 100 ))
        
        # Apply the cap
        chmod 644 "$policy/scaling_max_freq" 2>/dev/null
        echo "$target_val" > "$policy/scaling_max_freq"
        
        log_msg "CPU: $(basename $policy) capped at ${target_val}kHz"
    fi
done

sync
log_msg "SYSTEM: Optimization Complete."
