#!/system/bin/sh

# --- PATHS ---
MOD_DIR="/data/adb/modules/MTK_AI"
BB="$MOD_DIR/busybox"
EXTERNAL_CFG="/sdcard/MTK_AI_Engine"
MAINTENANCE_PATH="$MOD_DIR/MTK_AI/AI_MODE/auto_frequency/auto_frequency.sh"
BOOSTER_PATH="$MOD_DIR/MTK_AI/AI_MODE/auto_frequency/booster.sh"
LIMITER_PATH="$MOD_DIR/MTK_AI/AI_MODE/auto_frequency/cpulimiter.sh"
TIMEOUT=10

# Temp files
TS_FILE="/data/local/tmp/touch_ts"
PID_FILE="/data/local/tmp/touch_monitor.pid"
GETEVENT_PID_FILE="/data/local/tmp/touch_getevent.pid"

# Prefer system date; fallback to busybox
DATE_CMD="/system/bin/date"
[ -x "$DATE_CMD" ] || DATE_CMD="$BB date"
[ -x "$DATE_CMD" ] || { echo "❌ date not found"; exit 1; }

# --- UTILITIES ---
log() { echo "[$($DATE_CMD '+%T')] $*" >&2; }
die() { log "❌ $*"; exit 1; }

run_cmd() {
    [ -f "$1" ] && {
        log "⚙️ Running: ${1##*/}"
        sh "$1" >/dev/null 2>&1 &
    }
}

# --- CLEANUP (critical for no background persistence) ---
cleanup() {
    log "🧹 Cleaning up..."

    # Kill getevent by PID (safe)
    if [ -f "$GETEVENT_PID_FILE" ]; then
        GETEVENT_PID=$(cat "$GETEVENT_PID_FILE")
        if kill -0 "$GETEVENT_PID" 2>/dev/null; then
            kill "$GETEVENT_PID" 2>/dev/null
            wait "$GETEVENT_PID" 2>/dev/null
        fi
        rm -f "$GETEVENT_PID_FILE"
    fi

    rm -f "$TS_FILE" "$PID_FILE"
    exit 0
}

trap cleanup TERM INT EXIT

# --- 1. FIND TOUCHSCREEN ---
find_touchscreen() {
    for dev in /dev/input/event*; do
        if getevent -p "$dev" 2>/dev/null | grep -q "0035"; then
            echo "$dev"
            return 0
        fi
    done
    return 1
}

TOUCH_DEV=$(find_touchscreen) || die "Touchscreen not found"
log "🎯 Touch device: $TOUCH_DEV"

# --- 2. KILL OLD INSTANCE (safe PID-based) ---
if [ -f "$PID_FILE" ]; then
    OLD_PID=$(cat "$PID_FILE")
    if [ -n "$OLD_PID" ] && kill -0 "$OLD_PID" 2>/dev/null; then
        log "🔁 Stopping old instance (PID $OLD_PID)..."
        kill "$OLD_PID" 2>/dev/null
        sleep 0.3
    fi
    rm -f "$PID_FILE"
fi

# --- 3. LAUNCH GETEVENT (background, PID-tracked) ---
# Your original logic — proven to work ✅
(
    getevent "$TOUCH_DEV" | while read -r _; do
        # Only update timestamp on *any* event — your method works!
        echo "$($DATE_CMD +%s)" > "$TS_FILE"
    done
) &
GETEVENT_PID=$!
echo "$GETEVENT_PID" > "$GETEVENT_PID_FILE"

# Ensure getevent is alive
sleep 0.2
kill -0 "$GETEVENT_PID" 2>/dev/null || die "getevent failed to start"

# --- 4. INITIAL STATE ---
echo "$($DATE_CMD +%s)" > "$TS_FILE"
STATE="PERF"
echo $$ > "$PID_FILE"

log "🚀 Touch Monitor Active (PID $$)"
log "⏳ Timeout: ${TIMEOUT}s"

# --- 5. MAIN LOOP ---
while true; do
    NOW=$($DATE_CMD +%s)
    # Cache timestamp (avoid repeated cat)
    [ -f "$TS_FILE" ] || echo "$NOW" > "$TS_FILE"
    LAST_TS=$(cat "$TS_FILE")

    # Safety: avoid negative/invalid elapsed
    ELAPSED=$((NOW - LAST_TS))
    [ "$ELAPSED" -lt 0 ] && ELAPSED=0

    if [ "$ELAPSED" -ge "$TIMEOUT" ]; then
        if [ "$STATE" = "PERF" ]; then
            STATE="SAVE"
            log "🔋 Switching to SAVE (idle $ELAPSED s)"
            [ -f "$EXTERNAL_CFG/enable_limiter" ] && run_cmd "$LIMITER_PATH"
        fi
    else
        if [ "$STATE" = "SAVE" ]; then
            STATE="PERF"
            log "⚡ Switching to PERF (touch after $ELAPSED s)"
            run_cmd "$BOOSTER_PATH"
        fi
    fi

    # Keep your UI (optional: comment out if running as service)
    printf "\r[Dev: %s] [Mode: %s] [Idle: %ds]   " "${TOUCH_DEV##*/}" "$STATE" "$ELAPSED"
    sleep 1
done