#!/system/bin/sh
# ULTRA FAST CHARGING OPTIMIZER - UNIVERSAL COMPATIBLE
# Compatible: Snapdragon | MediaTek | Exynos | Kirin | Unisoc
# Support: 10W to 120W All Charger Types
# Feature: Auto Cut-off 100% & Auto Resume 95%

# Check root
if [[ "$(id -u)" -ne 0 ]]; then
    echo "❌ No root permissions. Exiting."
    exit 1
fi

# System info
chipset=$(getprop ro.board.platform 2>/dev/null || getprop ro.chipname 2>/dev/null || echo "universal")
androidVersion=$(getprop ro.build.version.release)
batteryCapacity=$(cat /sys/class/power_supply/battery/capacity 2>/dev/null || echo "N/A")

echo "=========================================="
echo "⚡ ULTRA FAST CHARGING OPTIMIZER ⚡"
echo "=========================================="
echo "Chipset: $chipset"
echo "Battery: ${batteryCapacity}%"
echo "Android: $androidVersion"
echo "=========================================="

# Enhanced safe write with validation
safe_write() {
    if [[ -f "$1" ]]; then
        echo "$2" > "$1" 2>/dev/null && echo "✅ $1 = $2" || echo "⚠️ $1 (failed)"
    fi
}

# Advanced Chipset Detection
detect_chipset() {
    echo "🔍 Detecting Chipset Type..."
    
    # Snapdragon Detection
    if [[ "$chipset" == *"msm"* || "$chipset" == *"qcom"* || "$chipset" == *"sdm"* ]] || \
       grep -qi "qualcomm" /proc/cpuinfo 2>/dev/null || \
       [[ -d "/sys/class/kgsl/kgsl-3d0" ]]; then
        echo "✅ Snapdragon Detected"
        CHIPSET_TYPE="snapdragon"
        
    # MediaTek Detection  
    elif [[ "$chipset" == *"mt"* || "$chipset" == *"helio"* || "$chipset" == *"dimensity"* ]] || \
         grep -qi "mediatek" /proc/cpuinfo 2>/dev/null || \
         [[ -f "/proc/gpufreq/gpufreq_opp_freq" ]]; then
        echo "✅ MediaTek Detected"
        CHIPSET_TYPE="mediatek"
        
    # Exynos Detection
    elif [[ "$chipset" == *"exynos"* || "$chipset" == *"s5e"* ]] || \
         grep -qi "samsung" /proc/cpuinfo 2>/dev/null || \
         [[ -f "/sys/power/cpufreq_max_limit" ]]; then
        echo "✅ Exynos Detected"
        CHIPSET_TYPE="exynos"
        
    # Kirin Detection
    elif [[ "$chipset" == *"kirin"* || "$chipset" == *"hi"* ]] || \
         grep -qi "hisilicon" /proc/cpuinfo 2>/dev/null; then
        echo "✅ Kirin Detected"
        CHIPSET_TYPE="kirin"
        
    # Unisoc Detection
    elif [[ "$chipset" == *"sprd"* || "$chipset" == *"unisoc"* ]] || \
         grep -qi "sprd" /proc/cpuinfo 2>/dev/null; then
        echo "✅ Unisoc Detected"
        CHIPSET_TYPE="unisoc"
        
    else
        echo "⚡ Universal Chipset Mode"
        CHIPSET_TYPE="universal"
    fi
}

# Universal Charging Path Detection
find_charging_controllers() {
    echo "🔌 Scanning Charging Controllers..."
    
    CHARGE_PATHS=()
    
    # Common battery paths
    local battery_paths=(
        "/sys/class/power_supply/battery"
        "/sys/class/power_supply/main"
        "/sys/class/power_supply/bms"
        "/sys/class/qcom-battery"
        "/sys/class/mtk-battery"
        "/sys/class/sec/charger"
        "/sys/class/power_supply/charge"
    )
    
    # Common USB paths
    local usb_paths=(
        "/sys/class/power_supply/usb"
        "/sys/class/power_supply/usb1"
        "/sys/class/power_supply/usb2"
        "/sys/class/power_supply/ac"
        "/sys/class/power_supply/wireless"
    )
    
    for path in "${battery_paths[@]}" "${usb_paths[@]}"; do
        if [[ -d "$path" ]]; then
            CHARGE_PATHS+=("$path")
            echo "✅ Found: $path"
        fi
    done
    
    if [[ ${#CHARGE_PATHS[@]} -eq 0 ]]; then
        echo "❌ No charging controllers found!"
        exit 1
    fi
}

# Smart Charger Detection & Current Limit
detect_charger_type() {
    echo "🔋 Detecting Charger Type..."
    
    local usb_current=$(cat /sys/class/power_supply/usb/current_max 2>/dev/null || echo "0")
    local usb_voltage=$(cat /sys/class/power_supply/usb/voltage_max 2>/dev/null || echo "5000000")
    
    # Calculate available power
    local available_power=$(( (usb_current * usb_voltage) / 1000000 ))
    
    if [[ "$available_power" -gt 65 ]]; then
        echo "🚀 65W+ Super Charger Detected"
        MAX_CURRENT="5000000"
        MAX_VOLTAGE="20000000"
    elif [[ "$available_power" -gt 45 ]]; then
        echo "⚡ 45W+ Fast Charger Detected"
        MAX_CURRENT="4000000" 
        MAX_VOLTAGE="15000000"
    elif [[ "$available_power" -gt 25 ]]; then
        echo "💨 25W+ Quick Charger Detected"
        MAX_CURRENT="3000000"
        MAX_VOLTAGE="12000000"
    elif [[ "$available_power" -gt 15 ]]; then
        echo "🔌 15W+ Fast Charger Detected"
        MAX_CURRENT="2000000"
        MAX_VOLTAGE="9000000"
    else
        echo "🔋 Standard Charger (10W)"
        MAX_CURRENT="1500000"
        MAX_VOLTAGE="5000000"
    fi
    
    echo "💡 Available Power: ${available_power}W"
    echo "⚡ Max Current: $((MAX_CURRENT / 1000))mA"
    echo "🔋 Max Voltage: $((MAX_VOLTAGE / 1000))mV"
}

# Chipset-Specific Charging Optimization
optimize_chipset_charging() {
    echo "🎯 Applying Chipset-Specific Optimization..."
    
    case "$CHIPSET_TYPE" in
        "snapdragon")
            # Qualcomm Quick Charge
            safe_write "/sys/class/power_supply/battery/allow_hvdcp3" "1"
            safe_write "/sys/class/power_supply/battery/hvdcp_icl_max" "$MAX_CURRENT"
            safe_write "/sys/class/power_supply/usb/pd_allowed" "1"
            safe_write "/sys/class/qcom-battery/restricted_charging" "0"
            ;;
            
        "mediatek")
            # MediaTek Pump Express
            safe_write "/sys/class/power_supply/battery/allow_pe" "1"
            safe_write "/sys/class/power_supply/battery/allow_pe_plus" "1"
            safe_write "/proc/mtk_thermal_control/charging_current" "$MAX_CURRENT"
            safe_write "/sys/class/mtk-battery/current_limit" "$MAX_CURRENT"
            ;;
            
        "exynos")
            # Samsung Adaptive Fast Charging
            safe_write "/sys/class/power_supply/battery/afc_charge" "1"
            safe_write "/sys/class/power_supply/battery/afc_disable" "0"
            safe_write "/sys/class/sec/charger/afc_disable" "0"
            ;;
            
        "kirin")
            # Huawei SuperCharge
            safe_write "/sys/class/power_supply/battery/scp_charge" "1"
            safe_write "/sys/class/power_supply/battery/charge_control_limit" "0"
            ;;
            
        "unisoc")
            # Unisoc Smart Charge
            safe_write "/sys/class/power_supply/battery/constant_charge_current_max" "$MAX_CURRENT"
            safe_write "/sys/class/power_supply/battery/input_current_limit" "$MAX_CURRENT"
            ;;
            
        *)
            # Universal fallback
            safe_write "/sys/class/power_supply/battery/constant_charge_current_max" "$MAX_CURRENT"
            safe_write "/sys/class/power_supply/battery/input_current_limit" "$MAX_CURRENT"
            ;;
    esac
}

# Universal Charging Optimization
optimize_universal_charging() {
    echo "⚡ Applying Universal Charging Optimization..."
    
    for path in "${CHARGE_PATHS[@]}"; do
        echo "🎯 Optimizing: $path"
        
        # Remove charging restrictions
        safe_write "$path/restricted_charging" "0" 2>/dev/null
        safe_write "$path/input_suspend" "0" 2>/dev/null
        safe_write "$path/input_current_limited" "0" 2>/dev/null
        
        # Set maximum safe currents
        if [[ "$path" == *"battery"* ]] || [[ "$path" == *"main"* ]] || [[ "$path" == *"bms"* ]]; then
            safe_write "$path/constant_charge_current_max" "$MAX_CURRENT" 2>/dev/null
            safe_write "$path/input_current_limit" "$MAX_CURRENT" 2>/dev/null
            safe_write "$path/voltage_max" "$MAX_VOLTAGE" 2>/dev/null
        fi
        
        if [[ "$path" == *"usb"* ]] || [[ "$path" == *"ac"* ]]; then
            safe_write "$path/current_max" "$MAX_CURRENT" 2>/dev/null
            safe_write "$path/voltage_max" "$MAX_VOLTAGE" 2>/dev/null
            safe_write "$path/input_current_limit" "$MAX_CURRENT" 2>/dev/null
        fi
        
        # Enable fast charging features
        safe_write "$path/fastchg_current_max" "$MAX_CURRENT" 2>/dev/null
        safe_write "$path/allow_fast_charge" "1" 2>/dev/null
        safe_write "$path/enable_charging_limit" "0" 2>/dev/null
    done
}

# Intelligent Thermal Management
optimize_thermal_management() {
    echo "🌡️ Applying Intelligent Thermal Management..."
    
    # Increase thermal thresholds for better charging
    for thermal_zone in /sys/class/thermal/thermal_zone*; do
        if [[ -d "$thermal_zone" ]]; then
            local zone_type=$(cat "$thermal_zone/type" 2>/dev/null)
            
            case "$zone_type" in
                "battery"|"charger"|"usb_port")
                    # Increase thermal limits for charging components
                    for trip_point in "$thermal_zone"/trip_point_*_temp; do
                        if [[ -f "$trip_point" ]]; then
                            local current_temp=$(cat "$trip_point" 2>/dev/null)
                            if [[ -n "$current_temp" ]] && [[ "$current_temp" -lt 50000 ]]; then
                                local new_temp=$((current_temp + 5000))
                                safe_write "$trip_point" "$new_temp" 2>/dev/null
                            fi
                        fi
                    done
                    ;;
            esac
        fi
    done
    
    # Chipset-specific thermal optimizations
    case "$CHIPSET_TYPE" in
        "snapdragon")
            safe_write "/sys/module/msm_thermal/parameters/enabled" "Y" 2>/dev/null
            safe_write "/sys/module/msm_thermal/core_control/enabled" "1" 2>/dev/null
            safe_write "/sys/module/msm_thermal/parameters/limit_temp_degC" "50" 2>/dev/null
            ;;
        "mediatek")
            safe_write "/proc/mtk_thermal_control/enabled" "1" 2>/dev/null
            safe_write "/proc/mtk_thermal_control/temp_limit" "50000" 2>/dev/null
            ;;
    esac
}

# Adaptive Current Control
apply_adaptive_current() {
    echo "🔄 Applying Adaptive Current Control..."
    
    local batt_temp=$(cat /sys/class/power_supply/battery/temp 2>/dev/null)
    local capacity=$(cat /sys/class/power_supply/battery/capacity 2>/dev/null)
    
    if [[ -n "$batt_temp" ]]; then
        local temp_c=$((batt_temp / 10))
        echo "🌡️ Battery Temperature: ${temp_c}°C"
        
        # Adaptive current based on temperature
        if [[ "$temp_c" -gt 50 ]]; then
            echo "🚨 Critical Temperature! Minimal Charging"
            CURRENT_LIMIT="500000"
        elif [[ "$temp_c" -gt 45 ]]; then
            echo "⚠️ High Temperature! Reduced Charging"
            CURRENT_LIMIT=$((MAX_CURRENT / 2))
        elif [[ "$temp_c" -gt 40 ]]; then
            echo "🌡️ Warm Temperature! Moderate Charging"
            CURRENT_LIMIT=$((MAX_CURRENT * 3 / 4))
        else
            echo "✅ Normal Temperature! Full Speed Charging"
            CURRENT_LIMIT="$MAX_CURRENT"
        fi
    else
        CURRENT_LIMIT="$MAX_CURRENT"
    fi
    
    # Apply adaptive current limit
    for path in "${CHARGE_PATHS[@]}"; do
        if [[ "$path" == *"battery"* ]] || [[ "$path" == *"main"* ]]; then
            safe_write "$path/constant_charge_current_max" "$CURRENT_LIMIT" 2>/dev/null
            safe_write "$path/input_current_limit" "$CURRENT_LIMIT" 2>/dev/null
        fi
    done
    
    echo "⚡ Adaptive Current: $((CURRENT_LIMIT / 1000))mA"
}

# Smart Battery Protection - Auto Cut-off 100% & Resume 95%
setup_smart_battery_protection() {
    echo "🛡️ Setting Up Smart Battery Protection..."
    
    # Enable battery protection features
    safe_write "/sys/class/power_supply/battery/charge_control_limit" "100" 2>/dev/null
    safe_write "/sys/class/power_supply/battery/charge_disable" "0" 2>/dev/null
    
    # Set charging limits for battery health
    safe_write "/sys/class/power_supply/battery/step_charging_enabled" "1" 2>/dev/null
    safe_write "/sys/class/power_supply/battery/sw_jeita_enabled" "1" 2>/dev/null
    
    echo "✅ Smart Protection: Auto Cut-off 100%, Resume 95%"
}

# Auto Cut-off & Resume System
start_auto_cutoff_resume() {
    echo "🔋 Starting Auto Cut-off & Resume System..."
    echo "💡 System will:"
    echo "   • Cut-off charging at 100%"
    echo "   • Auto-resume when battery drops to 95%"
    echo "   • Protect battery health long-term"
    
    local last_action="charging"
    
    while true; do
        local capacity=$(cat /sys/class/power_supply/battery/capacity 2>/dev/null)
        local status=$(cat /sys/class/power_supply/battery/status 2>/dev/null)
        
        if [[ -n "$capacity" ]]; then
            # Auto cut-off at 100%
            if [[ "$capacity" -ge 100 ]] && [[ "$status" == "Charging" ]]; then
                if [[ "$last_action" != "cutoff" ]]; then
                    echo "🛑 Battery 100% - Cutting off charging"
                    safe_write "/sys/class/power_supply/battery/input_suspend" "1" 2>/dev/null
                    safe_write "/sys/class/power_supply/usb/current_max" "0" 2>/dev/null
                    last_action="cutoff"
                fi
            
            # Auto resume at 95%  
            elif [[ "$capacity" -le 95 ]] && [[ "$status" != "Charging" ]]; then
                if [[ "$last_action" != "charging" ]]; then
                    echo "🔋 Battery 95% - Resuming charging"
                    safe_write "/sys/class/power_supply/battery/input_suspend" "0" 2>/dev/null
                    safe_write "/sys/class/power_supply/usb/current_max" "$MAX_CURRENT" 2>/dev/null
                    
                    # Re-apply charging optimization
                    optimize_universal_charging
                    last_action="charging"
                fi
            fi
            
            # Show status
            if [[ "$last_action" == "cutoff" ]]; then
                echo "💤 Monitoring: ${capacity}% (Charging Cut-off)"
            else
                echo "⚡ Monitoring: ${capacity}% (Charging Active)"
            fi
        fi
        
        sleep 10  # Check every 10 seconds
    done
}

# Real-time Charging Monitor
start_charging_monitor() {
    echo "📊 Starting Real-time Charging Monitor..."
    
    # Run auto cut-off in background
    start_auto_cutoff_resume &
    local cutoff_pid=$!
    
    echo "=========================================="
    echo "🔋 CHARGING STATUS MONITOR"
    echo "=========================================="
    
    while true; do
        local capacity=$(cat /sys/class/power_supply/battery/capacity 2>/dev/null || echo "N/A")
        local status=$(cat /sys/class/power_supply/battery/status 2>/dev/null || echo "N/A")
        local health=$(cat /sys/class/power_supply/battery/health 2>/dev/null || echo "N/A")
        local temp=$(cat /sys/class/power_supply/battery/temp 2>/dev/null)
        local voltage=$(cat /sys/class/power_supply/battery/voltage_now 2>/dev/null)
        local current=$(cat /sys/class/power_supply/battery/current_now 2>/dev/null)
        
        clear
        echo "🔋 Capacity: ${capacity}% | Status: $status"
        echo "❤️ Health: $health | Chipset: $CHIPSET_TYPE"
        
        if [[ -n "$temp" ]]; then
            local temp_c=$((temp / 10))
            echo "🌡️ Temperature: ${temp_c}°C"
        fi
        
        if [[ -n "$voltage" ]] && [[ -n "$current" ]]; then
            local voltage_v=$((voltage / 1000))
            local current_ma=$((current / 1000))
            local power_mw=$(( (voltage_v * current_ma) / 1000 ))
            
            echo "⚡ Voltage: ${voltage_v}mV | Current: ${current_ma}mA"
            echo "💨 Power: ${power_mw}mW (≈$((power_mw / 1000))W)"
            
            # Charger type indicator
            if [[ "$power_mw" -gt 65000 ]]; then
                echo "🚀 SUPER CHARGER: 65W+"
            elif [[ "$power_mw" -gt 45000 ]]; then
                echo "⚡ FAST CHARGER: 45W+"
            elif [[ "$power_mw" -gt 25000 ]]; then
                echo "💨 QUICK CHARGER: 25W+"
            elif [[ "$power_mw" -gt 15000 ]]; then
                echo "🔌 FAST CHARGER: 15W+"
            else
                echo "🔋 STANDARD CHARGER: 10W"
            fi
        fi
        
        echo ""
        echo "🛡️ Auto Protection: Cut-off 100% | Resume 95%"
        echo "⏳ Monitoring... Press Ctrl+C to stop"
        echo "=========================================="
        
        sleep 3
    done
    
    # Cleanup background process if needed
    kill $cutoff_pid 2>/dev/null
}

# Main Execution
main() {
    echo "🚀 Starting Ultra Fast Charging Optimizer..."
    
    # Detection phase
    detect_chipset
    find_charging_controllers
    detect_charger_type
    
    # Optimization phase
    optimize_chipset_charging
    optimize_universal_charging
    optimize_thermal_management
    apply_adaptive_current
    setup_smart_battery_protection
    
    # Monitoring phase with auto cut-off
    start_charging_monitor
}

# Run main function
main